<?php
// user.php - Read-only user panel (WhatsApp removed for user; improved light-theme gradient)
// PHP 7.3 compatible. Uses index.php?action=list and index.php?action=search for data.
$config = require __DIR__ . '/config.php';
if (!empty($config['timezone'])) date_default_timezone_set($config['timezone']);
?>
<!doctype html>
<html lang="en">
<head>
<meta charset="utf-8">
<meta name="viewport" content="width=device-width,initial-scale=1">
<title>Udhaar — User View</title>
<style>
:root{
  --bg:#06080a; --bg2:#05080a;
  --panel:#0f1517; --muted:#9aa4b2; --accent1:#14e6d1; --accent2:#2fb7ff; --text:#eaf2f6; --gold:#f6c85f;
}
/* improved light gradient - smooth greenish gradient (no white band) */
:root.light{
  --bg:#f0fff7; /* pale green */
  --bg2:#d8fff0; /* slightly deeper */
  --panel:#ffffff; --muted:#6b7280; --accent1:#06b6b4; --accent2:#06a0ff; --text:#071018; --gold:#f6c85f;
}
*{box-sizing:border-box;font-family:Inter,system-ui,-apple-system,Segoe UI,Roboto}
html,body{height:100%}
body{
  margin:0;
  background: linear-gradient(180deg, var(--bg) 0%, var(--bg2) 100%);
  color:var(--text);
  min-height:100vh;
  -webkit-font-smoothing:antialiased;
  -moz-osx-font-smoothing:grayscale;
  background-attachment: fixed;
}
.container{max-width:980px;margin:0 auto;padding:18px 16px 80px;position:relative}

/* header */
.header-wrap{padding-top:56px;position:relative}
.brand h1{margin:0;font-size:1.6rem}
.top-center{position:absolute;left:50%;top:6px;transform:translateX(-50%);z-index:20}
.admin-btn{background:linear-gradient(90deg,var(--accent1),var(--accent2));border:0;padding:8px 14px;border-radius:18px;color:#012;cursor:pointer;box-shadow:0 10px 26px rgba(0,0,0,0.35);font-weight:600}

/* theme slide toggle */
.header{display:flex;align-items:center;justify-content:space-between;gap:12px;flex-wrap:wrap}
.header-controls{display:flex;align-items:center;gap:10px}
.theme-toggle { width:64px; height:36px; border-radius:999px; background:rgba(255,255,255,0.04); border:1px solid rgba(255,255,255,0.04); display:inline-flex; align-items:center; padding:4px; cursor:pointer; position:relative; transition:background .18s, box-shadow .18s; }
.theme-toggle .track { position:absolute; inset:4px; left:4px; right:4px; top:4px; bottom:4px; border-radius:999px; }
.theme-toggle .knob { width:28px; height:28px; border-radius:50%; background:var(--panel); display:flex; align-items:center; justify-content:center; transform:translateX(0); transition:transform .22s cubic-bezier(.2,.9,.3,1), box-shadow .18s; box-shadow:0 6px 18px rgba(0,0,0,0.4); }
.theme-toggle.active { background:linear-gradient(90deg,var(--accent1),var(--accent2)); box-shadow:0 8px 28px rgba(47,183,255,0.08); }
.theme-toggle.active .knob { transform:translateX(28px); box-shadow:0 10px 30px rgba(0,0,0,0.45); }
.theme-toggle svg { display:block; width:16px; height:16px; }
.theme-toggle .icon { color:var(--gold); filter:drop-shadow(0 3px 12px rgba(246,200,95,0.14)); }
.theme-toggle.active .knob .icon { filter:drop-shadow(0 6px 20px rgba(246,200,95,0.22)); }

/* top controls */
.top-row{display:flex;align-items:center;gap:10px;margin-top:16px;flex-wrap:wrap}
.search{flex:1;min-width:220px;padding:12px 14px;border-radius:14px;border:0;background:var(--panel);color:var(--muted);outline:none;box-shadow:0 8px 26px rgba(0,0,0,0.45)}

/* list */
.list{margin-top:18px;padding-bottom:40px}

/* person card */
.person{background:var(--panel);padding:14px;border-radius:14px;margin-bottom:12px;box-shadow:0 10px 30px rgba(0,0,0,0.45);transition:transform .12s}
.person:hover{transform:translateY(-6px)}
.person-head{display:flex;align-items:center;gap:12px;cursor:pointer}
.avatar{width:56px;height:56px;border-radius:12px;background:linear-gradient(135deg,#072b34,#06262e);display:flex;align-items:center;justify-content:center;color:var(--accent1);font-weight:700;font-size:18px;flex-shrink:0;position:relative;overflow:hidden;box-shadow:0 8px 30px rgba(0,0,0,0.45)}
@keyframes permabounce{0%{transform:translateY(0)}50%{transform:translateY(-6px)}100%{transform:translateY(0)}}
.avatar.perm{animation:permabounce 2.4s ease-in-out infinite}

/* name + meta + unpaid total */
.namebox{flex:1;display:flex;flex-direction:column;min-width:0}
.namebox h3{margin:0;font-size:1.05rem;white-space:nowrap;overflow:hidden;text-overflow:ellipsis}
.namebox .meta{color:var(--muted);font-size:13px;margin-top:6px}
.namebox .total{color:var(--text);font-weight:700;margin-top:8px;font-size:14px}

/* actions: chevron only for user */
.actions{display:flex;align-items:center;gap:10px;flex-shrink:0}
.chev{width:18px;height:18px;opacity:0.85;transform:rotate(0deg);transition:transform .28s}
.chev.open{transform:rotate(180deg)}

/* entries collapsed */
.entries{ display:block; max-height:0; opacity:0; transform:translateY(-6px); overflow:hidden; transition: max-height .36s cubic-bezier(.2,.9,.3,1), opacity .28s, transform .28s; margin-top:12px }
.entries.open{ max-height:2000px; opacity:1; transform:none; }

/* entry as own box */
.entry-box{background:linear-gradient(180deg, rgba(255,255,255,0.02), transparent); border-radius:12px; padding:14px; margin-bottom:12px; display:flex; justify-content:space-between; align-items:center; box-shadow:0 6px 18px rgba(0,0,0,0.35);}
.entry-left{display:flex;flex-direction:column;min-width:0}
.entry-title{font-weight:700;font-size:1rem;line-height:1.15}
.entry-meta{font-size:13px;color:var(--muted);margin-top:8px}
.entry-actions{display:flex;align-items:center;gap:10px}

/* tag + liquid */
.tag{position:relative;padding:6px 12px;border-radius:999px;font-weight:800;font-size:13px;color:#042;overflow:hidden;min-width:64px;text-align:center;display:inline-flex;align-items:center;justify-content:center;box-shadow:0 8px 26px rgba(0,0,0,0.35)}
.tag-paid{background:linear-gradient(90deg, rgba(16,185,129,0.98), rgba(16,185,129,0.88)); color:#042612}
.tag-unpaid{background:linear-gradient(90deg, rgba(239,68,68,0.98), rgba(239,68,68,0.88)); color:#3b0a0a}
.tag .liquid{position:absolute;width:120px;height:80px;border-radius:50%;right:-26px;top:-18px;background:radial-gradient(circle at 30% 35%, rgba(255,255,255,0.22), rgba(255,255,255,0.04) 28%, transparent 48%),linear-gradient(90deg, rgba(255,255,255,0.02), rgba(255,255,255,0.00));transform:translateX(0) scale(0.9);opacity:0.12;mix-blend-mode:screen;animation:liquidMove 5.6s linear infinite}
@keyframes liquidMove{0%{transform:translateX(0) scale(.9)}50%{transform:translateX(-32px) scale(1.05)}100%{transform:translateX(0) scale(.9)}}
.particle{position:absolute;width:6px;height:6px;border-radius:50%;opacity:0;background:rgba(255,255,255,0.8);animation:particleFloat 3.5s linear infinite}
@keyframes particleFloat{0%{opacity:0;transform:translateY(6px) translateX(0) scale(.6)}10%{opacity:.18}50%{opacity:.26;transform:translateY(-14px) translateX(-8px) scale(1)}90%{opacity:.12}100%{opacity:0;transform:translateY(6px) translateX(6px) scale(.6)}}

/* modal & misc (only entry editor modal kept) */
.modal-backdrop{position:fixed;inset:0;background:rgba(2,6,11,0.5);display:none;align-items:center;justify-content:center;z-index:999}
.modal-backdrop.show{display:flex}
.modal{background:var(--panel);padding:18px;border-radius:14px;box-shadow:0 10px 30px rgba(0,0,0,0.45);width:92%;max-width:520px;transform:translateY(12px);opacity:0;transition:all .28s}
.modal-backdrop.show .modal{transform:none;opacity:1}
.input{width:100%;padding:10px;border-radius:10px;border:0;background:transparent;color:var(--text);margin:8px 0;outline:none;border:1px solid rgba(255,255,255,0.03)}
.row-between{display:flex;justify-content:space-between;gap:8px;align-items:center}
.save-btn{background:linear-gradient(90deg,var(--accent1),var(--accent2));border:0;padding:10px 14px;border-radius:10px;color:#022;cursor:pointer}
.cancel-btn{background:transparent;border:1px solid rgba(255,255,255,0.04);padding:10px 12px;border-radius:10px;color:var(--muted);cursor:pointer}

/* small helper */
.small-muted{color:var(--muted);font-size:13px}

/* responsive */
@media(max-width:720px){
  .top-center{top:6px}
  .header{align-items:flex-start}
  .avatar{width:48px;height:48px}
  .brand h1{font-size:1.3rem}
}
@media(max-width:480px){
  .person-head{align-items:flex-start}
  .namebox h3{white-space:normal;overflow:visible}
  .entry-box{flex-direction:column;align-items:flex-start}
  .entry-actions{width:100%;margin-top:10px;justify-content:flex-end}
}
</style>
</head>
<body>
<div class="container">
  <div class="header-wrap">
    <div class="top-center">
      <button class="admin-btn" onclick="window.location.href='index.php'">Open Admin Panel</button>
    </div>

    <div class="header">
      <div class="brand"><h1>Udhaar</h1></div>
      <div class="header-controls">
        <button id="themeToggle" class="theme-toggle" role="switch" aria-checked="false" aria-label="Toggle theme">
          <div class="track"></div>
          <div class="knob" id="knob"><svg class="icon" viewBox="0 0 24 24"></svg></div>
        </button>
      </div>
    </div>
  </div>

  <div class="top-row">
    <input id="search" class="search" placeholder="Search name or item...">
  </div>

  <div id="list" class="list" aria-live="polite"></div>
</div>

<!-- modal (entry editor - same as admin) -->
<div id="modalBackdrop" class="modal-backdrop" aria-hidden="true">
  <div class="modal" role="dialog" aria-modal="true">
    <h3 id="modalTitle">Add Entry</h3>
    <input id="modalItem" class="input" placeholder="Item name">
    <input id="modalDate" class="input" type="date">
    <input id="modalPrice" class="input" placeholder="Price (optional)" inputmode="numeric">
    <label style="display:block;color:var(--muted);margin-top:8px"><input id="modalPaid" type="checkbox"> Mark Paid</label>
    <div style="height:10px"></div>
    <div class="row-between">
      <button id="modalSave" class="save-btn">Save</button>
      <button id="modalCancel" class="cancel-btn">Cancel</button>
    </div>
  </div>
</div>

<script>
const $ = s => document.querySelector(s);

// API helper to call index.php endpoints
async function apiToAdmin(action, opts={}) {
  const method = opts.method || 'GET';
  let url = 'index.php?action=' + encodeURIComponent(action);
  if (method === 'GET' && opts.query) url += '&' + new URLSearchParams(opts.query).toString();
  const fetchOpts = { method };
  if (method !== 'GET') { fetchOpts.headers = {'Content-Type':'application/json'}; fetchOpts.body = JSON.stringify(opts.body || {}); }
  const res = await fetch(url, fetchOpts);
  return res.json();
}

function initials(name){ return name.split(' ').filter(Boolean).slice(0,2).map(s=>s[0].toUpperCase()).join(''); }
function formatDate(d){ try { return new Date(d).toLocaleDateString('en-IN'); } catch(e) { return d || ''; } }
function formatCurrency(n){ if (isNaN(n) || n === null) return '₹0'; return '₹' + Number(n).toFixed(0); }

// render list
async function load(q='') {
  const people = q ? await apiToAdmin('search',{ query:{ q } }) : await apiToAdmin('list');
  const list = $('#list'); list.innerHTML = '';
  if (!people || !people.length) { list.innerHTML = '<div style="color:var(--muted);margin-top:20px">No records.</div>'; return; }

  people.forEach(p => {
    const card = document.createElement('div'); card.className = 'person';
    const head = document.createElement('div'); head.className = 'person-head';

    const ava = document.createElement('div'); ava.className = 'avatar perm'; ava.textContent = initials(p.name);

    const namebox = document.createElement('div'); namebox.className = 'namebox';
    const h3 = document.createElement('h3'); h3.textContent = p.name;
    const meta = document.createElement('div'); meta.className = 'meta'; meta.textContent = (p.entries && p.entries.length) ? (p.entries.length + ' entries') : 'No entries';

    // unpaid total
    let unpaidTotal = 0;
    if (p.entries && p.entries.length) {
      p.entries.forEach(e=>{
        const priceRaw = e.price;
        const priceNum = parseFloat(String(priceRaw).replace(/[^0-9.\-]/g,''));
        if (!e.paid && !isNaN(priceNum)) unpaidTotal += priceNum;
      });
    }
    const totalDiv = document.createElement('div'); totalDiv.className = 'total'; totalDiv.textContent = 'Total (unpaid) = ' + formatCurrency(unpaidTotal);

    namebox.appendChild(h3); namebox.appendChild(meta); namebox.appendChild(totalDiv);

    // actions - only chevron for user UI
    const actions = document.createElement('div'); actions.className = 'actions';
    const chev = document.createElement('div'); chev.className='chev'; chev.innerHTML = '<svg viewBox="0 0 24 24" width="18" height="18"><path d="M6 9l6 6 6-6" stroke="currentColor" stroke-width="2" fill="none" stroke-linecap="round" stroke-linejoin="round"/></svg>';
    actions.appendChild(chev);

    head.appendChild(ava); head.appendChild(namebox); head.appendChild(actions);

    card.appendChild(head);

    const entriesDiv = document.createElement('div'); entriesDiv.className = 'entries';
    if (p.entries && p.entries.length) {
      p.entries.forEach(e => {
        const box = document.createElement('div'); box.className = 'entry-box';
        const left = document.createElement('div'); left.className = 'entry-left';
        const title = document.createElement('div'); title.className = 'entry-title'; title.textContent = e.itemName || '-';
        const meta2 = document.createElement('div'); meta2.className = 'entry-meta'; meta2.textContent = formatDate(e.date) + (e.price ? ' • ₹' + e.price : '');
        left.appendChild(title); left.appendChild(meta2);

        const right = document.createElement('div'); right.className = 'entry-actions';
        const tag = document.createElement('div'); tag.className = 'tag ' + (e.paid ? 'tag-paid' : 'tag-unpaid'); tag.textContent = e.paid ? 'Paid' : 'Unpaid';
        const liq = document.createElement('div'); liq.className = 'liquid'; tag.appendChild(liq);
        for (let i=1;i<=5;i++){ const pEl = document.createElement('div'); pEl.className = 'particle p'+i; tag.appendChild(pEl); }
        right.appendChild(tag);

        box.appendChild(left); box.appendChild(right);
        entriesDiv.appendChild(box);
      });
    } else {
      const none = document.createElement('div'); none.style.color = 'var(--muted)'; none.style.marginTop = '8px'; none.textContent = 'No entries';
      entriesDiv.appendChild(none);
    }

    card.appendChild(entriesDiv);
    list.appendChild(card);

    // toggle logic
    function toggle() {
      const isOpen = entriesDiv.classList.contains('open');
      if (isOpen) {
        entriesDiv.classList.remove('open');
        chev.querySelector('svg') && chev.querySelector('svg').classList.remove('open');
      } else {
        document.querySelectorAll('.entries.open').forEach(ed => { ed.classList.remove('open'); });
        document.querySelectorAll('.chev svg.open').forEach(c => c.classList.remove('open'));
        entriesDiv.classList.add('open');
        chev.querySelector('svg') && chev.querySelector('svg').classList.add('open');
        setTimeout(()=> card.scrollIntoView({ behavior:'smooth', block:'center' }), 160);
      }
    }

    head.addEventListener('click', toggle);
    ava.addEventListener('click', toggle);
  });
}

// search input
$('#search').addEventListener('input', (e)=> load(e.target.value));

// THEME toggle (same slider as admin)
const themeToggle = $('#themeToggle'), knob = $('#knob');
function moonSvg(){ return '<svg class=\"icon\" viewBox=\"0 0 24 24\" xmlns=\"http://www.w3.org/2000/svg\"><path fill=\"var(--gold)\" d=\"M21 12.79A9 9 0 1111.21 3 7 7 0 0021 12.79z\"/></svg>'; }
function sunSvg(){ return '<svg class=\"icon\" viewBox=\"0 0 24 24\" xmlns=\"http://www.w3.org/2000/svg\"><circle cx=\"12\" cy=\"12\" r=\"4\" fill=\"var(--gold)\"/><g stroke=\"var(--gold)\" stroke-width=\"1.2\" stroke-linecap=\"round\"><line x1=\"12\" y1=\"1.5\" x2=\"12\" y2=\"4.2\"/><line x1=\"12\" y1=\"19.8\" x2=\"12\" y2=\"22.5\"/><line x1=\"1.5\" y1=\"12\" x2=\"4.2\" y2=\"12\"/><line x1=\"19.8\" y1=\"12\" x2=\"22.5\" y2=\"12\"/><line x1=\"4\" y1=\"4\" x2=\"5.8\" y2=\"5.8\"/><line x1=\"18.2\" y1=\"18.2\" x2=\"20\" y2=\"20\"/><line x1=\"4\" y1=\"20\" x2=\"5.8\" y2=\"18.2\"/><line x1=\"18.2\" y1=\"5.8\" x2=\"20\" y2=\"4\"/></g></svg>'; }

function setTheme(light){
  if(light){
    document.documentElement.classList.add('light');
    themeToggle.classList.add('active');
    themeToggle.setAttribute('aria-checked','true');
    knob.innerHTML = sunSvg();
  } else {
    document.documentElement.classList.remove('light');
    themeToggle.classList.remove('active');
    themeToggle.setAttribute('aria-checked','false');
    knob.innerHTML = moonSvg();
  }
}
setTheme(false);
themeToggle.addEventListener('click', ()=> {
  const isLight = document.documentElement.classList.toggle('light');
  setTheme(isLight);
});
themeToggle.addEventListener('keydown', (e)=>{ if(e.key === 'Enter' || e.key === ' ') { e.preventDefault(); themeToggle.click(); } });

// initial render
load();
</script>
</body>
</html>