<?php
// index.php - Final Udhaar Manager (entries boxed + unpaid total shown)
// Modified minimally: added Extra Amount feature saving to extra.json and delete extra,
// plus UI CSS tweak for Add New Customer button appearance.
// Requires config.php with ['data_file','bot_token','chat_id','timezone'].

$config = require __DIR__ . '/config.php';
$dataFile = $config['data_file'] ?? __DIR__ . '/data.json';
$extraFile = __DIR__ . '/extra.json';
if (!empty($config['timezone'])) date_default_timezone_set($config['timezone']);

function read_db($file) {
    if (!file_exists($file)) {
        file_put_contents($file, json_encode(['people'=>[]], JSON_PRETTY_PRINT));
        @chmod($file, 0664);
    }
    $txt = @file_get_contents($file);
    $arr = @json_decode($txt, true);
    if (!is_array($arr)) $arr = ['people'=>[]];
    if (!isset($arr['people'])) $arr['people'] = [];
    return $arr;
}
function write_db($file, $data) {
    $tmp = $file . '.tmp';
    file_put_contents($tmp, json_encode($data, JSON_PRETTY_PRINT));
    rename($tmp, $file);
}
function uid() {
    try { return bin2hex(random_bytes(8)); } catch (Exception $e) { return substr(md5(uniqid('', true)),0,16); }
}

/* ---------- extras helpers ---------- */
function read_extras($file) {
    if (!file_exists($file)) {
        file_put_contents($file, json_encode(['extras'=>[]], JSON_PRETTY_PRINT));
        @chmod($file, 0664);
    }
    $txt = @file_get_contents($file);
    $arr = @json_decode($txt, true);
    if (!is_array($arr)) $arr = ['extras'=>[]];
    if (!isset($arr['extras'])) $arr['extras'] = [];
    return $arr;
}
function write_extras($file, $data) {
    $tmp = $file . '.tmp';
    file_put_contents($tmp, json_encode($data, JSON_PRETTY_PRINT));
    rename($tmp, $file);
}

/* ---------- API ---------- */
$action = isset($_REQUEST['action']) ? $_REQUEST['action'] : null;
if ($action) {
    header('Content-Type: application/json; charset=utf-8');
    $db = read_db($dataFile);

    if ($action === 'list') { echo json_encode($db['people']); exit; }

    if ($action === 'search') {
        $q = isset($_GET['q']) ? strtolower(trim($_GET['q'])) : '';
        if ($q === '') { echo json_encode($db['people']); exit; }
        $out = [];
        foreach ($db['people'] as $p) {
            $matchName = (strpos(strtolower($p['name']), $q) !== false);
            $matchEntries = array_values(array_filter($p['entries'], function($e) use($q){
                return stripos((string)($e['itemName'] ?? ''), $q) !== false;
            }));
            if ($matchName || count($matchEntries) > 0) {
                if (!$matchName) $p['entries'] = $matchEntries;
                $out[] = $p;
            }
        }
        echo json_encode($out);
        exit;
    }

    // add_person now accepts optional phone
    if ($action === 'add_person') {
        $input = json_decode(file_get_contents('php://input'), true);
        $name = trim($input['name'] ?? '');
        $phone = trim($input['phone'] ?? '');
        if ($name === '') { http_response_code(400); echo json_encode(['error'=>'Name required']); exit; }
        $person = ['id'=>uid(), 'name'=>$name, 'phone'=>$phone, 'entries'=>[]];
        array_unshift($db['people'], $person);
        write_db($dataFile, $db);
        echo json_encode($person); exit;
    }

    if ($action === 'rename_person') {
        $input = json_decode(file_get_contents('php://input'), true);
        $pid = $input['person_id'] ?? '';
        $newName = trim($input['name'] ?? '');
        foreach ($db['people'] as &$p) {
            if ($p['id'] === $pid) {
                if ($newName === '') { http_response_code(400); echo json_encode(['error'=>'Name required']); exit; }
                $p['name'] = $newName;
                write_db($dataFile, $db);
                echo json_encode(['ok'=>true]); exit;
            }
        }
        http_response_code(404); echo json_encode(['error'=>'Person not found']); exit;
    }

    // set phone for existing person if needed
    if ($action === 'set_phone') {
        $input = json_decode(file_get_contents('php://input'), true);
        $pid = $input['person_id'] ?? '';
        $phone = trim($input['phone'] ?? '');
        foreach ($db['people'] as &$p) {
            if ($p['id'] === $pid) {
                $p['phone'] = $phone;
                write_db($dataFile, $db);
                echo json_encode(['ok'=>true,'phone'=>$phone]); exit;
            }
        }
        http_response_code(404); echo json_encode(['error'=>'Person not found']); exit;
    }

    if ($action === 'delete_person') {
        $pid = $_GET['person_id'] ?? '';
        $before = count($db['people']);
        $db['people'] = array_values(array_filter($db['people'], function($p) use($pid){ return $p['id'] !== $pid; }));
        $after = count($db['people']);
        if ($after === $before) { http_response_code(404); echo json_encode(['error'=>'Person not found']); exit; }
        write_db($dataFile, $db);
        echo json_encode(['ok'=>true]); exit;
    }

    if ($action === 'add_entry') {
        $input = json_decode(file_get_contents('php://input'), true);
        $pid = $input['person_id'] ?? '';
        $item = $input['itemName'] ?? '';
        $date = $input['date'] ?? date('c');
        $price = (isset($input['price']) && $input['price'] !== '') ? $input['price'] : null;
        foreach ($db['people'] as &$p) {
            if ($p['id'] === $pid) {
                $entry = ['id'=>uid(),'itemName'=>$item,'date'=>$date,'price'=>$price,'paid'=>false,'paidAt'=>null];
                array_unshift($p['entries'],$entry);
                write_db($dataFile,$db);
                echo json_encode($entry); exit;
            }
        }
        http_response_code(404); echo json_encode(['error'=>'Person not found']); exit;
    }

    if ($action === 'edit_entry') {
        $input = json_decode(file_get_contents('php://input'), true);
        $pid = $input['person_id'] ?? '';
        $eid = $input['entry_id'] ?? '';
        foreach ($db['people'] as &$p) {
            if ($p['id'] === $pid) {
                foreach ($p['entries'] as &$e) {
                    if ($e['id'] === $eid) {
                        if (isset($input['itemName'])) $e['itemName'] = $input['itemName'];
                        if (isset($input['date'])) $e['date'] = $input['date'];
                        if (array_key_exists('price', $input)) $e['price'] = $input['price'] === '' ? null : $input['price'];
                        if (isset($input['paid'])) {
                            $newPaid = (bool)$input['paid'];
                            if ($newPaid && empty($e['paid'])) $e['paidAt'] = date('c');
                            if (!$newPaid) $e['paidAt'] = null;
                            $e['paid'] = $newPaid;
                        }
                        write_db($dataFile,$db);
                        echo json_encode($e); exit;
                    }
                }
            }
        }
        http_response_code(404); echo json_encode(['error'=>'Entry not found']); exit;
    }

    if ($action === 'delete_entry') {
        $pid = $_GET['person_id'] ?? '';
        $eid = $_GET['entry_id'] ?? '';
        foreach ($db['people'] as &$p) {
            if ($p['id'] === $pid) {
                $p['entries'] = array_values(array_filter($p['entries'], function($x) use($eid){ return $x['id'] !== $eid; }));
                write_db($dataFile,$db);
                echo json_encode(['ok'=>true]); exit;
            }
        }
        http_response_code(404); echo json_encode(['error'=>'Not found']); exit;
    }

    if ($action === 'send_now') {
        $msg = build_message($db['people']);
        $sent = send_telegram($config['bot_token'] ?? '', $config['chat_id'] ?? '', $msg);
        echo json_encode(['ok'=>$sent]); exit;
    }

    /* ---------- Extras endpoints ---------- */
    if ($action === 'list_extras') {
        $extr = read_extras($extraFile);
        echo json_encode($extr['extras']);
        exit;
    }

    if ($action === 'add_extra') {
        $input = json_decode(file_get_contents('php://input'), true);
        $name = trim($input['name'] ?? '');
        $person_id = $input['person_id'] ?? null;
        $amount = $input['amount'] ?? null;
        if ($name === '' || $amount === '' || $amount === null) { http_response_code(400); echo json_encode(['error'=>'Name and amount required']); exit; }
        $amount_num = floatval(preg_replace('/[^0-9.\-]/','',$amount));
        $extra = ['id'=>uid(),'person_id'=>$person_id?:null,'name'=>$name,'amount'=>($amount_num),'date'=>date('c')];
        $extr = read_extras($extraFile);
        array_unshift($extr['extras'], $extra);
        write_extras($extraFile, $extr);
        echo json_encode($extra);
        exit;
    }

    if ($action === 'extras_for') {
        $who = isset($_GET['name']) ? trim($_GET['name']) : '';
        $extr = read_extras($extraFile);
        if ($who === '') { echo json_encode($extr['extras']); exit; }
        $out = array_values(array_filter($extr['extras'], function($x) use($who){ return (stripos($x['name'],$who)!==false); }));
        echo json_encode($out); exit;
    }

    // NEW: delete extra by id
    if ($action === 'delete_extra') {
        $id = $_GET['extra_id'] ?? '';
        $extr = read_extras($extraFile);
        $before = count($extr['extras']);
        $extr['extras'] = array_values(array_filter($extr['extras'], function($x) use($id){ return $x['id'] !== $id; }));
        $after = count($extr['extras']);
        if ($after === $before) { http_response_code(404); echo json_encode(['error'=>'Not found']); exit; }
        write_extras($extraFile, $extr);
        echo json_encode(['ok'=>true]); exit;
    }

    echo json_encode(['error'=>'unknown action']);
    exit;
}

/* ---------- helpers ---------- */

function build_message($people) {
    $out = '';
    if (count($people) === 0) return "No udhaar entries.";
    foreach ($people as $p) {
        $out .= "<b>" . htmlspecialchars($p['name'], ENT_QUOTES, 'UTF-8') . "</b>\n";
        if (!count($p['entries'])) { $out .= "No entries\n\n"; continue; }
        $i = 1;
        foreach ($p['entries'] as $e) {
            $price = isset($e['price']) && $e['price'] !== null ? " | ₹" . htmlspecialchars($e['price']) : "";
            $paid = !empty($e['paid']) ? " ✅" : "";
            $date = isset($e['date']) ? date_format(date_create($e['date']),'d-m-Y') : '-';
            $out .= "{$i}. " . htmlspecialchars($e['itemName'] ?: '-') . " ({$date}){$price}{$paid}\n";
            $i++;
        }
        $out .= "\n";
    }
    return $out;
}
function send_telegram($bot_token,$chat_id,$text) {
    if (empty($bot_token) || empty($chat_id)) return false;
    $url = "https://api.telegram.org/bot{$bot_token}/sendMessage";
    $data = ['chat_id'=>$chat_id,'text'=>$text,'parse_mode'=>'HTML','disable_web_page_preview'=>true];
    $ch = curl_init($url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
    curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);
    $res = curl_exec($ch);
    curl_close($ch);
    $resp = json_decode($res, true);
    return isset($resp['ok']) && $resp['ok'] === true;
}
?>
<!doctype html>
<html lang="en">
<head>
<meta charset="utf-8">
<meta name="viewport" content="width=device-width,initial-scale=1">
<title>Udhaar Manager — Admin</title>
<style>
/* (kept your previous CSS — minor additions only for Add New Customer button & WA pill and extra modals) */

:root{
  /* dark theme variables */
  --bg: #06080a;
  --bg2: #05080a;
  --panel: #0f1517;
  --muted: #9aa4b2;
  --accent1: #14e6d1;
  --accent2: #2fb7ff;
  --text: #eaf2f6;
  --gold: #f6c85f;
}

/* light theme variables — adjust to avoid white band in gradient */
:root.light{
  --bg: #e8fdf6;
  --bg2: #d8fff0;
  --panel: #ffffff;
  --muted: #6b7280;
  --accent1: #06b6b4;
  --accent2: #06a0ff;
  --text: #071018;
  --gold: #f6c85f;
}

body{
  margin:0;
  background: linear-gradient(180deg, var(--bg) 0%, var(--bg2) 100%);
  color: var(--text);
  min-height: 100vh;
  background-attachment: fixed;
}
.container{max-width:980px;margin:0 auto;padding:18px 16px 80px;position:relative}
.header-wrap{padding-top:56px;position:relative}
.header{display:flex;align-items:center;justify-content:space-between;gap:12px;flex-wrap:wrap}
.brand{display:flex;flex-direction:column}
.brand h1{margin:0;font-size:1.6rem}
.header-controls{display:flex;align-items:center;gap:10px}
.top-center{position:absolute;left:50%;top:4px;transform:translateX(-50%);z-index:20}
.user-btn{background:linear-gradient(90deg,var(--accent1),var(--accent2));border:0;padding:8px 14px;border-radius:18px;color:#012;cursor:pointer;box-shadow:0 10px 26px rgba(0,0,0,0.35);font-weight:600}
.theme-toggle { width:64px; height:36px; border-radius:999px; background:rgba(255,255,255,0.04); border:1px solid rgba(255,255,255,0.04); display:inline-flex; align-items:center; padding:4px; cursor:pointer; position:relative; transition:background .18s, box-shadow .18s; }
.theme-toggle .track { position:absolute; inset:4px; left:4px; right:4px; top:4px; bottom:4px; border-radius:999px; }
.theme-toggle .knob { width:28px; height:28px; border-radius:50%; background:var(--panel); display:flex; align-items:center; justify-content:center; transform:translateX(0); transition:transform .22s cubic-bezier(.2,.9,.3,1), box-shadow .18s; box-shadow:0 6px 18px rgba(0,0,0,0.4); }
.theme-toggle.active { background:linear-gradient(90deg,var(--accent1),var(--accent2)); box-shadow:0 8px 28px rgba(47,183,255,0.08); }
.theme-toggle.active .knob { transform:translateX(28px); box-shadow:0 10px 30px rgba(0,0,0,0.45); }
.theme-toggle svg { display:block; width:16px; height:16px; }
.theme-toggle .icon { color:var(--gold); filter:drop-shadow(0 3px 12px rgba(246,200,95,0.14)); }
.theme-toggle.active .knob .icon { filter:drop-shadow(0 6px 20px rgba(246,200,95,0.22)); }

.top-row{display:flex;align-items:center;gap:10px;margin-top:16px;flex-wrap:wrap}
.search{flex:1;min-width:220px;padding:12px 14px;border-radius:14px;border:0;background:var(--panel);color:var(--muted);outline:none;box-shadow:0 8px 26px rgba(0,0,0,0.45)}
.send-btn{padding:10px 14px;border-radius:12px;border:0;background:linear-gradient(90deg,var(--accent1),var(--accent2));color:#012;cursor:pointer}

/* Extra & View Extra buttons */
.extra-btn{padding:10px 12px;border-radius:12px;border:0;background:#2ecc71;color:#052;cursor:pointer;font-weight:700;box-shadow:0 8px 24px rgba(46,204,113,0.06)}
.view-extra-btn{padding:10px 12px;border-radius:12px;border:0;background:linear-gradient(90deg,#ffd86b,#ffb54a);color:#012;cursor:pointer;font-weight:700;box-shadow:0 8px 24px rgba(255,200,110,0.06)}

.row{display:flex;gap:10px;align-items:center;margin-top:14px}
.add-input{flex:1;padding:12px;border-radius:12px;border:0;background:var(--panel);color:var(--text)}
.add-btn{width:64px;border-radius:12px;border:0;background:linear-gradient(90deg,var(--accent1),var(--accent2));color:#012;cursor:pointer}
.list{margin-top:18px;padding-bottom:40px}

/* FIXED: Add New Customer style restored (rounded gradient like original) */
.add-customer-btn { background: linear-gradient(90deg,var(--accent1),var(--accent2)); border:0; color:#012; padding:10px 14px; border-radius:18px; cursor:pointer; font-weight:700; box-shadow:0 10px 26px rgba(0,0,0,0.35); }

/* rest of CSS (kept same) */
.person{background:var(--panel);padding:14px;border-radius:14px;margin-bottom:12px;box-shadow:0 10px 30px rgba(0,0,0,0.45);transition:transform .12s}
.person:hover{transform:translateY(-6px)}
.person-head{display:flex;align-items:center;gap:12px;flex-wrap:wrap}
.avatar{width:56px;height:56px;border-radius:12px;background:linear-gradient(135deg,#072b34,#06262e);display:flex;align-items:center;justify-content:center;color:var(--accent1);font-weight:700;font-size:18px;cursor:pointer;flex-shrink:0;position:relative;overflow:hidden;box-shadow:0 8px 30px rgba(0,0,0,0.45)}
@keyframes permabounce{0%{transform:translateY(0)}50%{transform:translateY(-6px)}100%{transform:translateY(0)}}
.avatar.perm{animation:permabounce 2.4s ease-in-out infinite}
.namebox{flex:1;display:flex;flex-direction:column;min-width:0}
.namebox h3{margin:0;font-size:1.05rem;white-space:nowrap;overflow:hidden;text-overflow:ellipsis}
.namebox .meta{color:var(--muted);font-size:13px;margin-top:6px}
.namebox .total{color:var(--text);font-weight:700;margin-top:8px;font-size:14px}
.actions{display:flex;gap:8px;align-items:center;flex-shrink:0;white-space:nowrap}
.action-pill{display:inline-flex;align-items:center;justify-content:center;padding:8px 12px;border-radius:10px;border:0;background:rgba(255,255,255,0.03);color:var(--muted);cursor:pointer;font-weight:600}
.action-pill.gradient{background:linear-gradient(90deg,var(--accent1),var(--accent2));color:#012;box-shadow:0 8px 24px rgba(20,200,180,0.06)}
.action-pill.warn{background:linear-gradient(90deg,#ff7b6b,#ff4b4b);color:#fff;box-shadow:0 8px 24px rgba(255,80,60,0.06)}
.chev svg{opacity:.9;transition:transform .18s}
.chev svg.open{transform:rotate(180deg)}
.entries{ display:block; max-height:0; opacity:0; transform:translateY(-6px); overflow:hidden; transition: max-height .36s cubic-bezier(.2,.9,.3,1), opacity .28s, transform .28s; margin-top:12px }
.entries.open{ max-height:2000px; opacity:1; transform:none; }
.entry-box{background:linear-gradient(180deg, rgba(255,255,255,0.02), transparent); border-radius:12px; padding:14px; margin-bottom:12px; display:flex; justify-content:space-between; align-items:center; box-shadow:0 6px 18px rgba(0,0,0,0.35);}
.entry-left{display:flex;flex-direction:column;min-width:0}
.entry-title{font-weight:700;font-size:1rem;line-height:1.15}
.entry-meta{font-size:13px;color:var(--muted);margin-top:8px}
.entry-actions{display:flex;align-items:center;gap:10px}
.entry-actions .small-btn{padding:8px 10px;border-radius:10px;border:0;background:transparent;color:var(--muted);cursor:pointer}
.entry-actions .small-btn.warn{background:linear-gradient(90deg,#ff7b6b,#ff4b4b);color:#fff}
.tag{position:relative;padding:6px 12px;border-radius:999px;font-weight:800;font-size:13px;color:#042;overflow:hidden;min-width:64px;text-align:center;display:inline-flex;align-items:center;justify-content:center;box-shadow:0 8px 26px rgba(0,0,0,0.35)}
.tag-paid{background:linear-gradient(90deg, rgba(16,185,129,0.98), rgba(16,185,129,0.88)); color:#042612}
.tag-unpaid{background:linear-gradient(90deg, rgba(239,68,68,0.98), rgba(239,68,68,0.88)); color:#3b0a0a}
.tag .liquid{position:absolute;width:120px;height:80px;border-radius:50%;right:-26px;top:-18px;background:radial-gradient(circle at 30% 35%, rgba(255,255,255,0.22), rgba(255,255,255,0.04) 28%, transparent 48%),linear-gradient(90deg, rgba(255,255,255,0.02), rgba(255,255,255,0.00));transform:translateX(0) scale(0.9);opacity:0.12;mix-blend-mode:screen;animation:liquidMove 5.6s linear infinite}
@keyframes liquidMove{0%{transform:translateX(0) scale(.9)}50%{transform:translateX(-32px) scale(1.05)}100%{transform:translateX(0) scale(.9)}}
.particle{position:absolute;width:6px;height:6px;border-radius:50%;opacity:0;background:rgba(255,255,255,0.8);animation:particleFloat 3.5s linear infinite}
@keyframes particleFloat{0%{opacity:0;transform:translateY(6px) translateX(0) scale(.6)}10%{opacity:.18}50%{opacity:.26;transform:translateY(-14px) translateX(-8px) scale(1)}90%{opacity:.12}100%{opacity:0;transform:translateY(6px) translateX(6px) scale(.6)}}

/* WhatsApp pill style */
.wa-pill { display:inline-flex; align-items:center; gap:8px; padding:8px 12px; border-radius:12px; background:linear-gradient(90deg,#12b246,#25d366); color:#fff; font-weight:700; cursor:pointer; border:0; box-shadow:0 8px 20px rgba(18,178,70,0.12); }
.wa-pill svg { width:16px; height:16px; display:block; }

/* modal & misc */
.modal-backdrop{position:fixed;inset:0;background:rgba(2,6,11,0.5);display:none;align-items:center;justify-content:center;z-index:999}
.modal-backdrop.show{display:flex}
.modal{background:var(--panel);padding:18px;border-radius:14px;box-shadow:0 10px 30px rgba(0,0,0,0.45);width:92%;max-width:420px;transform:translateY(12px);opacity:0;transition:all .28s}
.modal-backdrop.show .modal{transform:none;opacity:1}
.input{width:100%;padding:10px;border-radius:10px;border:0;background:transparent;color:var(--text);margin:8px 0;outline:none;border:1px solid rgba(255,255,255,0.03)}
.row-between{display:flex;justify-content:space-between;gap:8px;align-items:center}
.save-btn{background:linear-gradient(90deg,var(--accent1),var(--accent2));border:0;padding:10px 14px;border-radius:10px;color:#022;cursor:pointer}
.cancel-btn{background:transparent;border:1px solid rgba(255,255,255,0.04);padding:10px 12px;border-radius:10px;color:var(--muted);cursor:pointer}
#toast{position:fixed;left:50%;transform:translateX(-50%);bottom:28px;background:rgba(0,0,0,0.7);color:#fff;padding:10px 14px;border-radius:10px;display:none;z-index:9999}

/* extra list styles inside modal */
.extras-group { margin-bottom:12px; }
.extras-item { display:flex; justify-content:space-between; padding:8px 10px; border-radius:8px; background:rgba(255,255,255,0.02); margin-bottom:6px; }

@media(max-width:720px){
  .top-center{top:6px}
  .header{align-items:flex-start}
  .avatar{width:48px;height:48px}
  .action-pill{padding:7px 10px;font-size:14px}
  .brand h1{font-size:1.3rem}
}
@media(max-width:480px){
  .person-head{align-items:flex-start}
  .namebox h3{white-space:normal;overflow:visible}
  .actions{width:100%;order:3;justify-content:flex-end;margin-top:8px}
  .chev{order:2;margin-left:6px}
  .entry-box{flex-direction:column;align-items:flex-start}
  .entry-actions{width:100%;margin-top:10px;justify-content:flex-end}
}
</style>
</head>
<body>
<div class="container">
  <div class="header-wrap">
    <div class="top-center">
      <button class="user-btn" onclick="window.location.href='user.php'">Open User Panel</button>
    </div>

    <div class="header">
      <div class="brand">
        <h1>Udhaar Manager</h1>
      </div>

      <div class="header-controls">
        <button id="themeToggle" class="theme-toggle" role="switch" aria-checked="false" aria-label="Toggle theme">
          <div class="track"></div>
          <div class="knob" id="knob"><svg class="icon" viewBox="0 0 24 24"></svg></div>
        </button>
      </div>
    </div>
  </div>

  <div class="top-row">
    <input id="search" class="search" placeholder="Search name or item...">
    <button id="sendNow" class="send-btn">Send Now</button>

    <!-- NEW: Extra amount + View extra buttons -->
    <button id="extraBtn" class="extra-btn" title="Add extra amount">Extra Amount</button>
    <button id="viewExtraBtn" class="view-extra-btn" title="View extras">View Extra</button>
  </div>

  <!-- replaced inline fields with single Add New Customer button -->
  <div class="row" style="margin-top:14px">
    <button id="openAddCustomer" class="add-customer-btn">Add New Customer</button>
  </div>

  <div id="list" class="list" aria-live="polite"></div>
</div>

<!-- Add New Customer Modal -->
<div id="addCustomerBackdrop" class="modal-backdrop" aria-hidden="true">
  <div class="modal" role="dialog" aria-modal="true">
    <h3 id="addCustomerTitle">Add New Customer</h3>
    <input id="custName" class="input" placeholder="Customer name">
    <input id="custPhone" class="input" placeholder="WhatsApp number (optional) e.g. +9198...">
    <div class="row-between" style="margin-top:10px">
      <div>
        <button id="addCustomerSave" class="save-btn">Save</button>
        <button id="addCustomerCancel" class="cancel-btn">Cancel</button>
      </div>
    </div>
  </div>
</div>

<!-- WhatsApp Modal (preview + save phone) -->
<div id="waBackdrop" class="modal-backdrop" aria-hidden="true">
  <div class="modal" role="dialog" aria-modal="true">
    <h3 id="waTitle">WhatsApp — Send Summary</h3>
    <div style="color:var(--muted);font-size:14px;margin-bottom:8px">Enter number to send (optional). Save to remember for next time.</div>
    <input id="waPhone" class="input" placeholder="+9198..." inputmode="tel">
    <div style="font-size:13px;color:var(--muted);margin:6px 0">Message preview:</div>
    <textarea id="waPreview" class="input" style="height:140px;resize:vertical"></textarea>
    <div class="row-between" style="margin-top:10px">
      <div>
        <button id="waSave" class="save-btn">Save</button>
        <button id="waCancel" class="cancel-btn">Cancel</button>
      </div>
      <div>
        <button id="waSend" class="save-btn">Send Summary</button>
      </div>
    </div>
  </div>
</div>

<!-- modal for Extra Amount -->
<div id="extraBackdrop" class="modal-backdrop" aria-hidden="true">
  <div class="modal" role="dialog" aria-modal="true">
    <h3>Add Extra Amount</h3>
    <label style="color:var(--muted)">Select customer (or type new)</label>
    <select id="extraPersonSelect" class="input" style="padding:10px"></select>
    <input id="extraNameInput" class="input" placeholder="Or type name manually (optional)">
    <input id="extraAmount" class="input" placeholder="Amount e.g. 250" inputmode="numeric">
    <div style="height:10px"></div>
    <div class="row-between">
      <button id="extraSave" class="save-btn">Save Extra</button>
      <button id="extraCancel" class="cancel-btn">Cancel</button>
    </div>
  </div>
</div>

<!-- modal for View Extras -->
<div id="viewExtraBackdrop" class="modal-backdrop" aria-hidden="true">
  <div class="modal" role="dialog" aria-modal="true">
    <h3>Extras list</h3>
    <input id="viewExtraFilter" class="input" placeholder="Filter by name...">
    <div id="extrasList" style="max-height:340px;overflow:auto;margin-top:8px"></div>
    <div style="height:10px"></div>
    <div style="text-align:right"><button id="viewExtraClose" class="save-btn">Close</button></div>
  </div>
</div>

<!-- entry add/edit modal -->
<div id="modalBackdrop" class="modal-backdrop" aria-hidden="true">
  <div class="modal" role="dialog" aria-modal="true">
    <h3 id="modalTitle">Add Entry</h3>
    <input id="modalItem" class="input" placeholder="Item name">
    <input id="modalDate" class="input" type="date">
    <input id="modalPrice" class="input" placeholder="Price (optional)" inputmode="numeric">
    <label style="display:block;color:var(--muted);margin-top:8px"><input id="modalPaid" type="checkbox"> Mark Paid</label>
    <div style="height:10px"></div>
    <div class="row-between">
      <button id="modalSave" class="save-btn">Save</button>
      <button id="modalCancel" class="cancel-btn">Cancel</button>
    </div>
  </div>
</div>

<!-- confirm -->
<div id="confirmBackdrop" class="modal-backdrop" aria-hidden="true">
  <div class="modal" role="dialog" aria-modal="true">
    <h3 id="confirmTitle">Confirm</h3>
    <div id="confirmText" style="color:var(--muted);margin:12px 0">Are you sure?</div>
    <div style="display:flex;justify-content:flex-end;gap:8px">
      <button id="confirmNo" class="cancel-btn">Cancel</button>
      <button id="confirmYes" class="save-btn">OK</button>
    </div>
  </div>
</div>

<div id="toast" role="status" aria-live="polite"></div>

<script>
/* JS: keep all existing behaviors, add Add-Customer modal and Extra modals integration. */

// helper
const $ = s => document.querySelector(s);
async function apiFetch(action, opts={}) {
  const method = opts.method || 'GET';
  let url = '?action=' + encodeURIComponent(action);
  if (method === 'GET' && opts.query) url += '&' + new URLSearchParams(opts.query).toString();
  const fetchOpts = { method };
  if (method !== 'GET') { fetchOpts.headers = {'Content-Type':'application/json'}; fetchOpts.body = JSON.stringify(opts.body || {}); }
  const res = await fetch(url, fetchOpts);
  return res.json();
}
function initials(name){ return name.split(' ').filter(Boolean).slice(0,2).map(s=>s[0].toUpperCase()).join(''); }
function formatDate(d){ try { return new Date(d).toLocaleDateString('en-IN'); } catch(e){ return d||''; } }
function showToast(txt, t=1600){ const to = $('#toast'); to.textContent = txt; to.style.display='block'; clearTimeout(to._tm); to._tm=setTimeout(()=> to.style.display='none', t); }

// Build message for WA preview
function buildWAmsg(person){
    let msg = "Hi " + person.name + ",\nYour udhaar summary:\n";
    let i = 1, total = 0;
    (person.entries || []).forEach(e => {
        const date = e.date ? new Date(e.date).toLocaleDateString('en-IN') : '-';
        const price = (e.price !== null && e.price !== undefined && e.price !== '') ? (" • ₹" + e.price) : "";
        const paid = e.paid ? " (Paid)" : " (Unpaid)";
        msg += `${i}) ${e.itemName || '-'} (${date})${price}${paid}\n`;
        if(!e.paid && !isNaN(parseFloat(e.price))) total += parseFloat(e.price);
        i++;
    });
    msg += `\nTotal unpaid: ₹${total}\n\n— Udhaar Manager`;
    return msg;
}
function makeWA(phone, text){
    const num = (phone||'').replace(/\D/g,'');
    if (!num) return null;
    return 'https://wa.me/' + encodeURIComponent(num) + '?text=' + encodeURIComponent(text);
}

// render list
async function load(q='') {
  const people = q ? await apiFetch('search',{ query:{ q } }) : await apiFetch('list');
  const list = $('#list'); list.innerHTML = '';
  if (!people || !people.length) { list.innerHTML = '<div style="color:var(--muted);margin-top:20px">No names yet. Add someone above.</div>'; return; }

  people.forEach(p => {
    const card = document.createElement('div'); card.className = 'person';
    const head = document.createElement('div'); head.className = 'person-head';

    const ava = document.createElement('div'); ava.className = 'avatar perm'; ava.textContent = initials(p.name);

    const namebox = document.createElement('div'); namebox.className = 'namebox';
    const h3 = document.createElement('h3'); h3.textContent = p.name;
    const meta = document.createElement('div'); meta.className = 'meta'; meta.textContent = (p.entries && p.entries.length) ? (p.entries.length + ' entries') : 'No entries';

    // compute unpaid total
    let unpaidTotal = 0;
    if (p.entries && p.entries.length) {
      p.entries.forEach(e=>{
        const priceRaw = e.price;
        const priceNum = parseFloat(String(priceRaw).replace(/[^0-9.\-]/g,''));
        if (!e.paid && !isNaN(priceNum)) unpaidTotal += priceNum;
      });
    }
    const totalDiv = document.createElement('div'); totalDiv.className = 'total';
    totalDiv.textContent = 'Total (unpaid) = ₹' + (isNaN(unpaidTotal)?0:unpaidTotal);

    namebox.appendChild(h3); namebox.appendChild(meta); namebox.appendChild(totalDiv);

    // right side: WhatsApp pill + chevron
    const rightWrap = document.createElement('div'); rightWrap.style.display='flex'; rightWrap.style.alignItems='center'; rightWrap.style.gap='10px';

    // WhatsApp pill (green)
    const waButton = document.createElement('button'); waButton.className = 'wa-pill';
    waButton.innerHTML = '<svg viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg"><path fill="#fff" d="M20.52 3.48A11.94 11.94 0 0012 0C5.373 0 .001 5.373 0 12c0 2.116.558 4.175 1.614 6.012L0 24l6.233-1.608A11.94 11.94 0 0012 24c6.627 0 12-5.373 12-12 0-3.2-1.247-6.195-3.48-8.52zM12 21.6a9.6 9.6 0 01-4.95-1.35l-.35-.2-3.7.96.98-3.6-.22-.36A9.6 9.6 0 1121.6 12 9.6 9.6 0 0112 21.6z"/><path fill="#25D366" d="M17.1 14.1l-1.2-.36a1.2 1.2 0 00-1.03.26l-.65.56a7.8 7.8 0 01-3.6-3.6l.56-.65c.24-.24.34-.58.26-1.02l-.36-1.2A1.2 1.2 0 009.9 6.9L8.3 7.2c-.32.08-.58.34-.66.66-.9 3.9 1.7 7.4 5.6 6.5.32-.08.58-.34.66-.66l.3-1.6c.04-.34-.18-.66-.5-.78z"/></svg><span>WhatsApp</span>';
    waButton.addEventListener('click', (ev)=> {
        ev.stopPropagation();
        openWAModal(p);
    });

    const chev = document.createElement('div'); chev.className='chev'; chev.innerHTML = '<svg viewBox="0 0 24 24" width="18" height="18"><path d="M6 9l6 6 6-6" stroke="currentColor" stroke-width="2" fill="none" stroke-linecap="round" stroke-linejoin="round"/></svg>';

    rightWrap.appendChild(waButton);
    rightWrap.appendChild(chev);

    head.appendChild(ava); head.appendChild(namebox); head.appendChild(rightWrap);
    card.appendChild(head);

    // entries area
    const entriesDiv = document.createElement('div'); entriesDiv.className = 'entries';
    if (p.entries && p.entries.length) {
      p.entries.forEach(e => {
        const box = document.createElement('div'); box.className = 'entry-box';
        const left = document.createElement('div'); left.className = 'entry-left';
        const title = document.createElement('div'); title.className = 'entry-title'; title.textContent = e.itemName || '-';
        const meta2 = document.createElement('div'); meta2.className = 'entry-meta'; meta2.textContent = formatDate(e.date) + (e.price ? ' • ₹' + e.price : '');
        left.appendChild(title); left.appendChild(meta2);

        const right = document.createElement('div'); right.className = 'entry-actions';
        const tag = document.createElement('div'); tag.className = 'tag ' + (e.paid ? 'tag-paid' : 'tag-unpaid'); tag.textContent = e.paid ? 'Paid' : 'Unpaid';
        const liq = document.createElement('div'); liq.className = 'liquid'; tag.appendChild(liq);
        for (let i=1;i<=5;i++){ const pEl = document.createElement('div'); pEl.className = 'particle p'+i; tag.appendChild(pEl); }

        const edit = document.createElement('button'); edit.className = 'small-btn'; edit.textContent = 'Edit';
        edit.addEventListener('click', (ev)=> { ev.stopPropagation(); openModal('edit', p.id, e.id); });
        const del = document.createElement('button'); del.className = 'small-btn warn'; del.textContent = 'Del';
        del.addEventListener('click', (ev)=> { ev.stopPropagation(); confirmAction('Delete this entry?', async ()=> {
          await fetch('?action=delete_entry&person_id=' + encodeURIComponent(p.id) + '&entry_id=' + encodeURIComponent(e.id));
          showToast('Deleted'); load($('#search').value);
        });});

        right.appendChild(tag); right.appendChild(edit); right.appendChild(del);
        box.appendChild(left); box.appendChild(right);
        entriesDiv.appendChild(box);
      });
    } else {
      const none = document.createElement('div'); none.style.color = 'var(--muted)'; none.style.marginTop = '8px'; none.textContent = 'No entries'; entriesDiv.appendChild(none);
    }
    card.appendChild(entriesDiv);

    // actions row (add entry, rename, delete person)
    const actions = document.createElement('div'); actions.className = 'actions';
    const addBtn = document.createElement('button'); addBtn.className = 'action-pill gradient'; addBtn.textContent = '+ Entry';
    addBtn.addEventListener('click', (ev)=> { ev.stopPropagation(); openModal('add', p.id); });
    const renameBtn = document.createElement('button'); renameBtn.className = 'action-pill'; renameBtn.textContent = 'Rename';
    renameBtn.addEventListener('click', (ev)=> { ev.stopPropagation(); openRename(p.id, p.name); });
    const delBtn = document.createElement('button'); delBtn.className = 'action-pill warn'; delBtn.textContent = 'Delete';
    delBtn.addEventListener('click', (ev)=> { ev.stopPropagation(); confirmAction('Delete this name and all its entries?', async ()=>{ await fetch('?action=delete_person&person_id=' + encodeURIComponent(p.id)); showToast('Deleted'); load($('#search').value); }); });

    actions.appendChild(addBtn); actions.appendChild(renameBtn); actions.appendChild(delBtn);
    card.appendChild(actions);

    // toggle entries when clicking head (but not when clicking action buttons)
    head.addEventListener('click', (ev)=> {
      if (ev.target.closest('.action-pill') || ev.target.closest('.wa-pill') || ev.target.closest('.entry-actions') || ev.target.closest('.small-btn')) return;
      const isOpen = entriesDiv.classList.contains('open');
      document.querySelectorAll('.entries.open').forEach(ed => ed.classList.remove('open'));
      document.querySelectorAll('.chev svg.open').forEach(c => c.classList.remove('open'));
      if (!isOpen) { entriesDiv.classList.add('open'); chev.querySelector('svg') && chev.querySelector('svg').classList.add('open'); setTimeout(()=> card.scrollIntoView({ behavior:'smooth', block:'center' }), 160); }
      else { entriesDiv.classList.remove('open'); chev.querySelector('svg') && chev.querySelector('svg').classList.remove('open'); }
    });

    list.appendChild(card);
  });
}

// Add customer modal interactions
$('#openAddCustomer').addEventListener('click', ()=> {
  $('#custName').value=''; $('#custPhone').value=''; $('#addCustomerBackdrop').classList.add('show');
});
$('#addCustomerCancel').addEventListener('click', ()=> { $('#addCustomerBackdrop').classList.remove('show'); });
$('#addCustomerSave').addEventListener('click', async ()=> {
  const name = $('#custName').value.trim(); const phone = $('#custPhone').value.trim();
  if (!name) { showToast('Name required'); return; }
  await apiFetch('add_person', { method:'POST', body:{ name, phone }});
  $('#addCustomerBackdrop').classList.remove('show');
  load();
  showToast('Customer added');
});

// WhatsApp modal logic
let currentPersonForWA = null;
function openWAModal(person) {
    currentPersonForWA = person;
    $('#waTitle').textContent = 'Send WhatsApp — ' + person.name;
    $('#waPhone').value = person.phone || '';
    $('#waPreview').value = buildWAmsg(person);
    $('#waBackdrop').classList.add('show');
}
$('#waCancel').addEventListener('click', ()=> { $('#waBackdrop').classList.remove('show'); currentPersonForWA = null; });
$('#waSave').addEventListener('click', async ()=> {
    if (!currentPersonForWA) return;
    const phone = $('#waPhone').value.trim();
    await apiFetch('set_phone', { method:'POST', body:{ person_id: currentPersonForWA.id, phone }});
    showToast('Phone saved');
    $('#waBackdrop').classList.remove('show');
    load($('#search').value);
});
$('#waSend').addEventListener('click', ()=> {
    if (!currentPersonForWA) return;
    const phone = $('#waPhone').value.trim();
    const msg = $('#waPreview').value.trim() || buildWAmsg(currentPersonForWA);
    const link = makeWA(phone, msg);
    if (!link) { showToast('Enter a valid phone number'); return; }
    window.open(link, '_blank');
    $('#waBackdrop').classList.remove('show');
});

// reuse your modal logic for entries
function openRename(pid, old) {
  $('#modalTitle').textContent = 'Rename';
  $('#modalItem').value = old;
  $('#modalDate').style.display = 'none';
  $('#modalPrice').style.display = 'none';
  $('#modalPaid').parentElement.style.display = 'none';
  modalState = { mode: 'rename', pid };
  $('#modalBackdrop').classList.add('show');
}
const modalBackdrop = $('#modalBackdrop'); let modalState = null;
function openModal(mode, pid, eid) {
  modalState = { mode, pid, eid };
  $('#modalTitle').textContent = mode === 'add' ? 'Add Entry' : 'Edit Entry';
  $('#modalItem').value = '';
  $('#modalDate').style.display = 'block';
  $('#modalPrice').style.display = 'block';
  $('#modalPaid').parentElement.style.display = 'block';
  $('#modalDate').value = new Date().toISOString().slice(0,10);
  $('#modalPrice').value = '';
  $('#modalPaid').checked = false;
  if (mode === 'edit') {
    apiFetch('list').then(people=>{
      const p = people.find(x=>x.id===pid);
      if (!p) return;
      const e = (p.entries||[]).find(x=>x.id===eid);
      if (!e) return;
      $('#modalItem').value = e.itemName || '';
      $('#modalDate').value = (e.date||'').split('T')[0] || new Date().toISOString().slice(0,10);
      $('#modalPrice').value = e.price || '';
      $('#modalPaid').checked = !!e.paid;
    });
  }
  modalBackdrop.classList.add('show');
}
$('#modalCancel').addEventListener('click', ()=> {
  $('#modalBackdrop').classList.remove('show');
  $('#modalDate').style.display = 'block';
  $('#modalPrice').style.display = 'block';
  $('#modalPaid').parentElement.style.display = 'block';
  modalState = null;
});
$('#modalSave').addEventListener('click', async ()=> {
  if (modalState && modalState.mode === 'rename') {
    const newName = $('#modalItem').value.trim();
    if (!newName) { showToast('Name required'); return; }
    await apiFetch('rename_person', { method:'POST', body:{ person_id: modalState.pid, name: newName } });
    $('#modalBackdrop').classList.remove('show');
    load($('#search').value);
    showToast('Renamed');
    $('#modalDate').style.display = 'block';
    $('#modalPrice').style.display = 'block';
    $('#modalPaid').parentElement.style.display = 'block';
    modalState = null;
    return;
  }

  const item = $('#modalItem').value.trim();
  const date = $('#modalDate').value;
  const price = $('#modalPrice').value.trim();
  const paid = $('#modalPaid').checked;
  if (!item) { showToast('Item name required'); return; }
  if (modalState.mode === 'add') {
    await apiFetch('add_entry', { method:'POST', body:{ person_id: modalState.pid, itemName: item, date: date, price: price } });
    $('#modalBackdrop').classList.remove('show');
    load($('#search').value);
    showToast('Entry added'); return;
  }
  if (modalState.mode === 'edit') {
    await apiFetch('edit_entry', { method:'POST', body:{ person_id: modalState.pid, entry_id: modalState.eid, itemName: item, date: date, price: price, paid: paid } });
    $('#modalBackdrop').classList.remove('show');
    load($('#search').value);
    showToast('Updated'); return;
  }
});

function confirmAction(text, cb) {
  $('#confirmTitle').textContent = 'Confirm';
  $('#confirmText').textContent = text;
  $('#confirmBackdrop').classList.add('show');
  $('#confirmNo').onclick = ()=> { $('#confirmBackdrop').classList.remove('show'); };
  $('#confirmYes').onclick = async ()=> { $('#confirmBackdrop').classList.remove('show'); await cb(); };
}
$('#confirmNo').addEventListener('click', ()=> $('#confirmBackdrop').classList.remove('show'));

// add person quick via modal
// handled above via Add Customer

// search & send
$('#search').addEventListener('input', (e)=> load(e.target.value));
$('#sendNow').addEventListener('click', ()=> {
  confirmAction('Send summary now to Telegram?', async ()=> {
    const res = await apiFetch('send_now', { method:'POST' });
    showToast(res.ok ? 'Sent.' : 'Send failed');
  });
});

// Theme toggle (moon/sun golden)
const themeToggle = $('#themeToggle'), knob = $('#knob');
function moonSvg(){ return '<svg class=\"icon\" viewBox=\"0 0 24 24\" xmlns=\"http://www.w3.org/2000/svg\"><path fill=\"var(--gold)\" d=\"M21 12.79A9 9 0 1111.21 3 7 7 0 0021 12.79z\"/></svg>'; }
function sunSvg(){ return '<svg class=\"icon\" viewBox=\"0 0 24 24\" xmlns=\"http://www.w3.org/2000/svg\"><circle cx=\"12\" cy=\"12\" r=\"4\" fill=\"var(--gold)\"/><g stroke=\"var(--gold)\" stroke-width=\"1.2\" stroke-linecap=\"round\"><line x1=\"12\" y1=\"1.5\" x2=\"12\" y2=\"4.2\"/><line x1=\"12\" y1=\"19.8\" x2=\"12\" y2=\"22.5\"/><line x1=\"1.5\" y1=\"12\" x2=\"4.2\" y2=\"12\"/><line x1=\"19.8\" y1=\"12\" x2=\"22.5\" y2=\"12\"/></g></svg>'; }
function setTheme(light){
  if(light){
    document.documentElement.classList.add('light');
    themeToggle.classList.add('active');
    themeToggle.setAttribute('aria-checked','true');
    knob.innerHTML = sunSvg();
  } else {
    document.documentElement.classList.remove('light');
    themeToggle.classList.remove('active');
    themeToggle.setAttribute('aria-checked','false');
    knob.innerHTML = moonSvg();
  }
}
setTheme(false);
themeToggle.addEventListener('click', ()=> {
  const isLight = document.documentElement.classList.toggle('light'); setTheme(isLight);
});
themeToggle.addEventListener('keydown', (e)=>{ if(e.key === 'Enter' || e.key === ' ') { e.preventDefault(); themeToggle.click(); } });

// ---------- EXTRA UI logic ----------
const extraBackdrop = $('#extraBackdrop'), extraPersonSelect = $('#extraPersonSelect'), extraNameInput = $('#extraNameInput'), extraAmount = $('#extraAmount'), extraSave = $('#extraSave'), extraCancel = $('#extraCancel');
const viewExtraBackdrop = $('#viewExtraBackdrop'), extrasList = $('#extrasList'), viewExtraFilter = $('#viewExtraFilter'), viewExtraClose = $('#viewExtraClose');

async function populateExtraPersonSelect() {
  const people = await apiFetch('list');
  extraPersonSelect.innerHTML = '<option value="">-- choose existing --</option>';
  people.forEach(p => {
    const opt = document.createElement('option'); opt.value = p.id; opt.textContent = p.name; extraPersonSelect.appendChild(opt);
  });
}

$('#extraBtn').addEventListener('click', async ()=> {
  await populateExtraPersonSelect();
  extraNameInput.value = '';
  extraAmount.value = '';
  extraBackdrop.classList.add('show');
});
extraCancel.addEventListener('click', ()=> extraBackdrop.classList.remove('show'));

extraSave.addEventListener('click', async ()=> {
  const sel = extraPersonSelect.value;
  const typed = extraNameInput.value.trim();
  const name = typed || (extraPersonSelect.options[extraPersonSelect.selectedIndex] ? extraPersonSelect.options[extraPersonSelect.selectedIndex].text : '');
  const person_id = sel || null;
  const amount = extraAmount.value.trim();
  if (!name || !amount) { showToast('Name and amount required'); return; }
  try {
    const res = await apiFetch('add_extra', { method:'POST', body:{ name, person_id, amount } });
    if (res && res.id) {
      showToast('Extra saved');
      extraBackdrop.classList.remove('show');
      load($('#search').value);
    } else {
      showToast('Save failed');
    }
  } catch(e) { showToast('Save failed'); }
});

// Open extras view
$('#viewExtraBtn').addEventListener('click', async ()=> {
  await renderExtras();
  viewExtraBackdrop.classList.add('show');
});
viewExtraClose.addEventListener('click', ()=> viewExtraBackdrop.classList.remove('show'));
viewExtraFilter.addEventListener('input', ()=> renderExtras(viewExtraFilter.value.trim()));

async function renderExtras(filter='') {
  const all = await apiFetch('list_extras');
  const list = (filter? all.filter(x=> x.name.toLowerCase().includes(filter.toLowerCase())) : all);
  extrasList.innerHTML = '';
  if (!list.length) { extrasList.innerHTML = '<div style="color:var(--muted)">No extra amounts yet.</div>'; return; }
  // group by name
  const grouped = {};
  list.forEach(x => {
    const k = x.name;
    if (!grouped[k]) grouped[k] = [];
    grouped[k].push(x);
  });
  for (const name in grouped) {
    const group = grouped[name];
    const wrapper = document.createElement('div'); wrapper.className='extras-group';
    const title = document.createElement('div'); title.style.fontWeight='700'; title.style.marginBottom='6px'; title.textContent = name + ' (Total: ₹' + group.reduce((s,i)=> s + Number(i.amount),0) + ')';
    wrapper.appendChild(title);
    group.forEach(e=>{
      const r = document.createElement('div'); r.className='extras-item';
      const left = document.createElement('div'); left.textContent = formatDate(e.date) + ' • ₹' + (Number(e.amount)||0);
      const right = document.createElement('div');
      // Add delete button per extra
      const delBtn = document.createElement('button'); delBtn.className='cancel-btn'; delBtn.textContent='Delete';
      delBtn.style.padding='6px 8px'; delBtn.style.borderRadius='8px';
      delBtn.addEventListener('click', async ()=> {
        if (!confirm('Delete this extra?')) return;
        try {
          const res = await fetch('?action=delete_extra&extra_id=' + encodeURIComponent(e.id));
          const j = await res.json();
          if (j.ok) { showToast('Deleted'); renderExtras(viewExtraFilter.value.trim()); load($('#search').value); }
          else showToast('Delete failed');
        } catch(err){ showToast('Delete failed'); }
      });
      right.appendChild(delBtn);
      r.appendChild(left); r.appendChild(right);
      wrapper.appendChild(r);
    });
    extrasList.appendChild(wrapper);
  }
}

// helper: open extra modal prefilled for a name (WhatsApp pill triggers this)
function openExtraForName(name) {
  populateExtraPersonSelect().then(()=>{
    const opts = Array.from(extraPersonSelect.options);
    const found = opts.find(o => o.text === name);
    if (found) {
      extraPersonSelect.value = found.value;
      extraNameInput.value = '';
    } else {
      extraPersonSelect.value = '';
      extraNameInput.value = name;
    }
    extraAmount.value = '';
    extraBackdrop.classList.add('show');
  });
}

// initial load
load();
</script>
</body>
</html>