<?php
// cron.php
// Run this via cron once per day (recommended at 12:00 PM IST).
// Behavior:
// 1) If local time is 12:00 -> send full current data summary to Telegram
// 2) Always: delete entries where paid=true and paidAt >= 7 days
//    and if any entries are deleted, send a deletion report to Telegram.

// Load config
$config = require __DIR__ . '/config.php';
$dataFile = $config['data_file'] ?? __DIR__ . '/data.json';
$botToken = $config['bot_token'] ?? '';
$chatId = $config['chat_id'] ?? '';

if (!empty($config['timezone'])) {
    date_default_timezone_set($config['timezone']);
}

// ----------------- helpers -----------------
function read_db($file) {
    if (!file_exists($file)) {
        file_put_contents($file, json_encode(['people' => []], JSON_PRETTY_PRINT));
        @chmod($file, 0664);
    }
    $txt = @file_get_contents($file);
    $arr = @json_decode($txt, true);
    if (!is_array($arr)) $arr = ['people' => []];
    if (!isset($arr['people'])) $arr['people'] = [];
    return $arr;
}
function write_db($file, $data) {
    $tmp = $file . '.tmp';
    file_put_contents($tmp, json_encode($data, JSON_PRETTY_PRINT));
    rename($tmp, $file);
}
function send_telegram($bot_token, $chat_id, $text) {
    if (empty($bot_token) || empty($chat_id)) return false;
    $url = "https://api.telegram.org/bot{$bot_token}/sendMessage";
    $data = [
        'chat_id' => $chat_id,
        'text' => $text,
        'parse_mode' => 'HTML',
        'disable_web_page_preview' => true
    ];
    $ch = curl_init($url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
    curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);
    curl_setopt($ch, CURLOPT_TIMEOUT, 20);
    $res = curl_exec($ch);
    $err = curl_error($ch);
    curl_close($ch);
    if ($res === false) {
        file_put_contents(__DIR__ . '/cron.log', date('c') . " telegram_curl_error: {$err}\n", FILE_APPEND);
        return false;
    }
    $resp = @json_decode($res, true);
    return isset($resp['ok']) && $resp['ok'] === true;
}
function build_message_full($people) {
    $out = '';
    if (empty($people)) return "No udhaar entries.";
    foreach ($people as $p) {
        $out .= "<b>" . htmlspecialchars($p['name'], ENT_QUOTES, 'UTF-8') . "</b>\n";
        if (empty($p['entries'])) {
            $out .= "No entries\n\n";
            continue;
        }
        $i = 1;
        foreach ($p['entries'] as $e) {
            $price = (isset($e['price']) && $e['price'] !== null && $e['price'] !== '') ? " | ₹" . htmlspecialchars($e['price']) : "";
            $paid = !empty($e['paid']) ? " ✅" : "";
            $date = isset($e['date']) && $e['date'] ? date_format(date_create($e['date']), 'd-m-Y') : '-';
            $out .= "{$i}. " . htmlspecialchars($e['itemName'] ?: '-') . " ({$date}){$price}{$paid}\n";
            $i++;
        }
        $out .= "\n";
    }
    return $out;
}
function build_deleted_report($deletedEntries) {
    if (empty($deletedEntries)) return '';
    $out = "<b>Deleted paid entries (>=7 days)</b>\n";
    foreach ($deletedEntries as $d) {
        // d: ['person'=>..., 'item'=>..., 'price'=>..., 'paidAt'=>..., 'days'=>int]
        $price = ($d['price'] !== null && $d['price'] !== '') ? " | ₹" . htmlspecialchars($d['price']) : "";
        $out .= "<b>" . htmlspecialchars($d['person'], ENT_QUOTES, 'UTF-8') . "</b> — " .
                htmlspecialchars($d['item'] ?: '-', ENT_QUOTES, 'UTF-8') .
                " ({$d['paidAt']})" . $price . " — {$d['days']}d\n";
    }
    return $out;
}

// ----------------- main -----------------
$db = read_db($dataFile);

// 1) Send full daily summary at 12:00 local time
// CHECK: where is this in the file? -> This check is here:
$nowLocal = new DateTime();
$timeNow = $nowLocal->format('H:i'); // e.g. "12:00"
$sentSummary = false;
if ($timeNow === '12:00') {
    $msg = build_message_full($db['people']);
    $sentSummary = send_telegram($botToken, $chatId, $msg);
    file_put_contents(__DIR__ . '/cron.log', date('c') . " send_summary_status: " . ($sentSummary ? 'ok' : 'fail') . "\n", FILE_APPEND);
} else {
    // not 12:00; log only
    file_put_contents(__DIR__ . '/cron.log', date('c') . " send_summary_skipped (time {$timeNow})\n", FILE_APPEND);
}

// 2) Cleanup: delete entries where paid=true and paidAt >= 7 days
$now = new DateTime();
$changed = false;
$deletedEntries = []; // keep for report
foreach ($db['people'] as &$person) {
    if (empty($person['entries']) || !is_array($person['entries'])) {
        $person['entries'] = [];
        continue;
    }
    $newEntries = [];
    foreach ($person['entries'] as $entry) {
        if (!empty($entry['paid']) && !empty($entry['paidAt'])) {
            try {
                $paidAt = new DateTime($entry['paidAt']);
                $diff = $now->diff($paidAt);
                $days = (int)$diff->format('%a');
                if ($days >= 7) {
                    // mark for deletion and continue (do NOT add to newEntries)
                    $deletedEntries[] = [
                        'person' => $person['name'] ?? 'Unknown',
                        'item'   => $entry['itemName'] ?? '-',
                        'price'  => isset($entry['price']) ? $entry['price'] : null,
                        'paidAt' => $entry['paidAt'],
                        'days'   => $days
                    ];
                    $changed = true;
                    continue;
                }
            } catch (Exception $ex) {
                // parsing failed -> keep entry (safer)
                $newEntries[] = $entry;
                continue;
            }
        } else {
            // keep unpaid or entries missing paidAt
            $newEntries[] = $entry;
        }
    }
    $person['entries'] = array_values($newEntries);
}
unset($person);

if ($changed) {
    write_db($dataFile, $db);
    file_put_contents(__DIR__ . '/cron.log', date('c') . " cleanup_done deleted_count=" . count($deletedEntries) . "\n", FILE_APPEND);

    // Build deletion report and send to Telegram
    $delMsg = build_deleted_report($deletedEntries);
    if ($delMsg !== '') {
        $sentDel = send_telegram($botToken, $chatId, $delMsg);
        file_put_contents(__DIR__ . '/cron.log', date('c') . " send_delete_report: " . ($sentDel ? 'ok' : 'fail') . "\n", FILE_APPEND);
    }
} else {
    file_put_contents(__DIR__ . '/cron.log', date('c') . " cleanup_nothing\n", FILE_APPEND);
}

// optional: rotate cron.log if too big
$logFile = __DIR__ . '/cron.log';
if (file_exists($logFile) && filesize($logFile) > 5 * 1024 * 1024) {
    @rename($logFile, __DIR__ . '/cron.log.old');
    file_put_contents(__DIR__ . '/cron.log', date('c') . " log_rotated\n", FILE_APPEND);
}

echo "OK\n";